// State
let isRecording = false;
let recordingStartTime = null;
let durationInterval = null;
let currentSteps = [];

// Cached DOM elements
const el = {};

// Throttle helper
const throttle = (fn, delay) => {
  let last = 0;
  return (...args) => {
    const now = Date.now();
    if (now - last >= delay) {
      last = now;
      fn(...args);
    }
  };
};

// Initialize popup
function init() {
  // Cache all DOM queries upfront
  el.recordBtn = document.getElementById('recordBtn');
  el.recordBtnText = document.getElementById('recordBtnText');
  el.statusBadge = document.getElementById('statusBadge');
  el.statusText = document.getElementById('statusText');
  el.serverStatus = document.getElementById('serverStatus');
  el.serverText = document.getElementById('serverText');
  el.recordingInfo = document.getElementById('recordingInfo');
  el.stepCount = document.getElementById('stepCount');
  el.duration = document.getElementById('duration');
  el.stepsList = document.getElementById('stepsList');
  el.actionsSection = document.getElementById('actionsSection');
  el.saveBtn = document.getElementById('saveBtn');
  el.exportBtn = document.getElementById('exportBtn');
  el.sendToCliBtn = document.getElementById('sendToCliBtn');

  // Check recording status
  updateStatus();

  // Check server status
  updateServerStatus();

  // Setup event listeners
  el.recordBtn.addEventListener('click', handleRecordClick);
  el.saveBtn.addEventListener('click', handleSave);
  el.exportBtn.addEventListener('click', handleExport);
  el.sendToCliBtn.addEventListener('click', handleSendToCli);

  // Poll for updates - throttled to reduce CPU
  const throttledPoll = throttle(() => {
    if (isRecording) {
      updateSteps();
    }
    updateServerStatus();
  }, 2000);

  setInterval(throttledPoll, 2000);
}

// Update server status display
async function updateServerStatus() {
  try {
    const response = await chrome.runtime.sendMessage({
      type: 'GET_SERVER_STATUS'
    });

    if (response.actuallyRunning || response.serverRunning) {
      el.serverStatus.classList.add('online');
      el.serverStatus.classList.remove('offline');
      el.serverText.textContent = 'ONLINE';
    } else {
      el.serverStatus.classList.remove('online');
      el.serverStatus.classList.add('offline');
      el.serverText.textContent = 'OFFLINE';
    }
  } catch (err) {
    el.serverStatus.classList.remove('online');
    el.serverStatus.classList.add('offline');
    el.serverText.textContent = 'UNKNOWN';
  }
}

// Update recording status from background
async function updateStatus() {
  try {
    const response = await chrome.runtime.sendMessage({
      type: 'GET_RECORDING_STATUS'
    });

    isRecording = response.isRecording;

    if (isRecording) {
      recordingStartTime = response.startedAt;
      startRecordingUI();
      await updateSteps();
    } else {
      stopRecordingUI();
    }
  } catch (err) {
    // Silently fail - extension may be reloading
  }
}

// Handle record button click
async function handleRecordClick() {
  if (isRecording) {
    await stopRecording();
  } else {
    await startRecording();
  }
}

// Start recording
async function startRecording() {
  try {
    const response = await chrome.runtime.sendMessage({
      type: 'START_RECORDING'
    });

    if (response.ok) {
      isRecording = true;
      recordingStartTime = Date.now();
      currentSteps = [];
      startRecordingUI();
    } else {
      throw new Error(response.error || 'Failed to start recording');
    }
  } catch (err) {
    alert('Failed to start recording: ' + err.message);
  }
}

// Stop recording
async function stopRecording() {
  try {
    const response = await chrome.runtime.sendMessage({
      type: 'STOP_RECORDING'
    });

    if (response.ok) {
      isRecording = false;
      currentSteps = response.steps || [];
      stopRecordingUI();
      updateStepsList();

      if (currentSteps.length > 0) {
        el.actionsSection.style.display = 'flex';
      }
    } else {
      throw new Error(response.error || 'Failed to stop recording');
    }
  } catch (err) {
    alert('Failed to stop recording: ' + err.message);
  }
}

// Update UI to recording state
function startRecordingUI() {
  el.recordBtn.classList.add('recording');
  el.recordBtnText.textContent = 'STOP RECORDING';
  el.statusBadge.classList.add('recording');
  el.statusText.textContent = 'RECORDING';
  el.recordingInfo.style.display = 'block';
  el.actionsSection.style.display = 'none';

  // Start duration timer
  durationInterval = setInterval(updateDuration, 1000);
}

// Update UI to stopped state
function stopRecordingUI() {
  el.recordBtn.classList.remove('recording');
  el.recordBtnText.textContent = 'START RECORDING';
  el.statusBadge.classList.remove('recording');
  el.statusText.textContent = 'READY';
  el.recordingInfo.style.display = 'none';

  // Stop duration timer
  if (durationInterval) {
    clearInterval(durationInterval);
    durationInterval = null;
  }
}

// Update duration display
function updateDuration() {
  if (!recordingStartTime) return;

  const elapsed = Date.now() - recordingStartTime;
  const seconds = Math.floor(elapsed / 1000);
  const minutes = Math.floor(seconds / 60);
  const secs = seconds % 60;

  el.duration.textContent = `${String(minutes).padStart(2, '0')}:${String(secs).padStart(2, '0')}`;
}

// Update steps from background
async function updateSteps() {
  try {
    const response = await chrome.runtime.sendMessage({
      type: 'GET_RECORDED_STEPS'
    });

    if (response.ok) {
      currentSteps = response.steps || [];
      el.stepCount.textContent = currentSteps.length;
      updateStepsList();
    }
  } catch (err) {
    // Silently fail
  }
}

// Update steps list UI - optimized with DocumentFragment
function updateStepsList() {
  if (currentSteps.length === 0) {
    el.stepsList.innerHTML = '<div class="empty-state"><p>No steps recorded yet</p><p class="empty-state-hint">$ Click START RECORDING to begin</p></div>';
    return;
  }

  // Use DocumentFragment for batch DOM updates
  const fragment = document.createDocumentFragment();

  currentSteps.forEach((step, index) => {
    const timeStr = new Date(step.timestamp).toLocaleTimeString('en-US', { hour12: false });
    const details = formatStepDetails(step);

    const stepDiv = document.createElement('div');
    stepDiv.className = 'step-item';
    stepDiv.innerHTML = `<div class="step-header"><span class="step-number">${String(index + 1).padStart(2, '0')}</span><span class="step-type">${step.type.toUpperCase()}</span><span class="step-time">${timeStr}</span></div>${details ? `<div class="step-details">> ${details}</div>` : ''}`;

    fragment.appendChild(stepDiv);
  });

  // Single DOM update
  el.stepsList.textContent = '';
  el.stepsList.appendChild(fragment);
}

// Format step details for display
function formatStepDetails(step) {
  const { type, params, semantic } = step;

  switch (type) {
    case 'click':
      return semantic.text ? `"${semantic.text}"` : semantic.tag;
    case 'type':
      return `Value: "${params.value}"`;
    case 'select':
      return `Selected: "${params.selectedText || params.value}"`;
    case 'check':
      return params.checked ? 'Checked' : 'Unchecked';
    case 'navigate':
      return `Page: "${params.title}"`;
    case 'scroll':
      return `Position: (${params.scrollX}, ${params.scrollY})`;
    default:
      return '';
  }
}

// Handle save workflow
async function handleSave() {
  if (currentSteps.length === 0) {
    alert('No steps to save');
    return;
  }

  try {
    const name = prompt('Enter workflow name:', 'My Workflow');
    if (!name) return;

    const description = prompt('Enter description (optional):', '');

    // Build workflow using WorkflowBuilder
    const builder = new WorkflowBuilder();
    const { workflow, domMap } = builder.buildFromRecording(
      currentSteps,
      name,
      description
    );

    // Save to IndexedDB
    const store = new DomMapStore();
    await store.open();
    await store.saveDomMap(domMap);
    await store.saveWorkflow(workflow);

    alert('Workflow saved successfully!');
  } catch (err) {
    alert('Failed to save workflow: ' + err.message);
  }
}

// Handle export workflow
async function handleExport() {
  if (currentSteps.length === 0) {
    alert('No steps to export');
    return;
  }

  try {
    const name = prompt('Enter workflow name:', 'My Workflow');
    if (!name) return;

    // Build workflow
    const builder = new WorkflowBuilder();
    const { workflow, domMap } = builder.buildFromRecording(currentSteps, name, '');

    // Create export object
    const exportData = {
      version: '1.0.0',
      workflow,
      domMap,
      exportedAt: Date.now()
    };

    // Download as JSON
    const blob = new Blob([JSON.stringify(exportData, null, 2)], {
      type: 'application/json'
    });
    const url = URL.createObjectURL(blob);

    const a = document.createElement('a');
    a.href = url;
    a.download = `${name.replace(/[^a-z0-9]/gi, '-').toLowerCase()}.json`;
    a.click();

    URL.revokeObjectURL(url);
  } catch (err) {
    alert('Failed to export workflow: ' + err.message);
  }
}

// Handle send to CLI - executes workflow via background script
async function handleSendToCli() {
  if (currentSteps.length === 0) {
    alert('No steps to send');
    return;
  }

  try {
    // Update button state
    el.sendToCliBtn.disabled = true;
    el.sendToCliBtn.textContent = '[EXEC] RUNNING...';

    // Build workflow
    const builder = new WorkflowBuilder();
    const { workflow, domMap } = builder.buildFromRecording(currentSteps, 'Quick Run', '');

    // Execute via background script
    const response = await chrome.runtime.sendMessage({
      type: 'EXECUTE_WORKFLOW',
      workflow,
      domMap,
      variables: {}
    });

    if (response.ok) {
      alert('Workflow executed successfully!');
    } else {
      throw new Error(response.error || 'Execution failed');
    }
  } catch (err) {
    // Offer fallback to clipboard
    const useClipboard = confirm(
      'Failed to execute: ' + err.message + '\n\n' +
      'Would you like to copy the workflow to clipboard instead?'
    );

    if (useClipboard) {
      const builder = new WorkflowBuilder();
      const { workflow, domMap } = builder.buildFromRecording(currentSteps, 'My Workflow', '');
      const payload = { version: '1.0.0', workflow, domMap, createdAt: Date.now() };
      await navigator.clipboard.writeText(JSON.stringify(payload, null, 2));
      alert('Workflow copied to clipboard!');
    }
  } finally {
    el.sendToCliBtn.disabled = false;
    el.sendToCliBtn.textContent = '[EXEC] SEND TO CLI';
  }
}

// Initialize when popup opens
init();
