/**
 * Popup Bridge Integration
 * Adds CLI connection functionality to the popup
 */

class PopupBridge {
    constructor() {
        this.bridge = new CLIBridge();
        this.isConnected = false;
        this.setupUI();
        this.setupEvents();
    }

    setupUI() {
        // Add CLI status indicator to popup
        const statusContainer = document.getElementById('cli-status') || this.createStatusElement();
        this.statusElement = statusContainer;
        this.updateStatusUI();

        // Add execute button
        const executeBtn = document.getElementById('execute-btn');
        if (executeBtn) {
            executeBtn.addEventListener('click', () => this.executeLastWorkflow());
        }
    }

    createStatusElement() {
        const container = document.createElement('div');
        container.id = 'cli-status';
        container.className = 'cli-status';
        container.innerHTML = `
            <span class="status-dot"></span>
            <span class="status-text">CLI: Disconnected</span>
            <button id="connect-cli-btn" class="btn-small">Connect</button>
        `;

        // Add to popup
        const header = document.querySelector('.header') || document.body;
        header.appendChild(container);

        // Connect button handler
        container.querySelector('#connect-cli-btn').addEventListener('click', () => {
            this.toggleConnection();
        });

        return container;
    }

    setupEvents() {
        this.bridge.on('connected', () => {
            this.isConnected = true;
            this.updateStatusUI();
        });

        this.bridge.on('disconnected', () => {
            this.isConnected = false;
            this.updateStatusUI();
        });

        this.bridge.on('error', (error) => {
            console.error('CLI Bridge error:', error);
            this.showNotification('CLI connection error', 'error');
        });
    }

    updateStatusUI() {
        const dot = this.statusElement.querySelector('.status-dot');
        const text = this.statusElement.querySelector('.status-text');
        const btn = this.statusElement.querySelector('#connect-cli-btn');

        if (this.isConnected) {
            dot.className = 'status-dot connected';
            text.textContent = 'CLI: Connected';
            btn.textContent = 'Disconnect';
        } else {
            dot.className = 'status-dot disconnected';
            text.textContent = 'CLI: Disconnected';
            btn.textContent = 'Connect';
        }
    }

    async toggleConnection() {
        if (this.isConnected) {
            this.bridge.disconnect();
        } else {
            try {
                await this.bridge.connect();
                this.showNotification('Connected to CLI', 'success');
            } catch (error) {
                this.showNotification('Failed to connect to CLI. Is it running?', 'error');
            }
        }
    }

    async executeLastWorkflow() {
        // Get last recorded workflow from storage
        const result = await chrome.storage.local.get(['lastWorkflow', 'lastDomMap']);

        if (!result.lastWorkflow) {
            this.showNotification('No workflow recorded yet', 'warning');
            return;
        }

        if (!this.isConnected) {
            this.showNotification('Connect to CLI first', 'warning');
            return;
        }

        try {
            // Sync DOM map first
            if (result.lastDomMap) {
                await this.bridge.syncDomMap(result.lastDomMap);
            }

            // Execute workflow
            this.showNotification('Executing workflow...', 'info');
            const execResult = await this.bridge.executeWorkflow(result.lastWorkflow);

            if (execResult.success) {
                this.showNotification('Workflow completed!', 'success');
            } else {
                this.showNotification(`Workflow failed: ${execResult.error}`, 'error');
            }

        } catch (error) {
            this.showNotification(`Execution error: ${error.message}`, 'error');
        }
    }

    showNotification(message, type = 'info') {
        // Simple notification - could be enhanced
        const notification = document.createElement('div');
        notification.className = `notification notification-${type}`;
        notification.textContent = message;
        document.body.appendChild(notification);

        setTimeout(() => {
            notification.remove();
        }, 3000);
    }
}

// Initialize when popup loads
document.addEventListener('DOMContentLoaded', () => {
    window.popupBridge = new PopupBridge();
});
