#!/usr/bin/env python3
"""
Eversale Native Messaging Host

This script is called by Chrome to start/stop the CLI bridge server.
It communicates with the Chrome extension via stdin/stdout using Chrome's
Native Messaging protocol.

Native Messaging Protocol:
- Messages are JSON with a 4-byte length prefix (little-endian)
- Chrome sends: {"action": "start"} or {"action": "stop"}
- Host responds: {"success": true, "port": 9876} or {"error": "..."}
"""

import asyncio
import json
import struct
import sys
import os
import signal
import subprocess
import time
import threading
from pathlib import Path

# Configuration
BRIDGE_PORT = 9876
BRIDGE_HOST = "localhost"

# Global state
bridge_process = None
bridge_task = None
server_running = False


def get_message():
    """Read a message from Chrome (stdin)"""
    try:
        # Read 4-byte length prefix
        raw_length = sys.stdin.buffer.read(4)
        if not raw_length:
            return None

        # Unpack length (little-endian unsigned int)
        message_length = struct.unpack('<I', raw_length)[0]

        # Read message
        message = sys.stdin.buffer.read(message_length).decode('utf-8')
        return json.loads(message)
    except Exception as e:
        log_error(f"Error reading message: {e}")
        return None


def send_message(message):
    """Send a message to Chrome (stdout)"""
    try:
        encoded = json.dumps(message).encode('utf-8')
        # Write 4-byte length prefix
        sys.stdout.buffer.write(struct.pack('<I', len(encoded)))
        # Write message
        sys.stdout.buffer.write(encoded)
        sys.stdout.buffer.flush()
    except Exception as e:
        log_error(f"Error sending message: {e}")


def log_error(msg):
    """Log errors to a file for debugging"""
    log_path = Path.home() / ".eversale" / "native_host.log"
    log_path.parent.mkdir(parents=True, exist_ok=True)
    with open(log_path, 'a') as f:
        f.write(f"{time.strftime('%Y-%m-%d %H:%M:%S')} - {msg}\n")


def is_port_in_use(port):
    """Check if port is already in use"""
    import socket
    with socket.socket(socket.AF_INET, socket.SOCK_STREAM) as s:
        try:
            s.bind((BRIDGE_HOST, port))
            return False
        except OSError:
            return True


def find_python():
    """Find Python executable"""
    # Try common Python paths
    candidates = [
        sys.executable,
        "python3",
        "python",
        "/usr/bin/python3",
        "/usr/local/bin/python3",
        str(Path.home() / ".eversale" / "venv" / "bin" / "python"),
        str(Path.home() / ".eversale" / "venv" / "Scripts" / "python.exe"),
    ]

    for candidate in candidates:
        try:
            result = subprocess.run(
                [candidate, "--version"],
                capture_output=True,
                timeout=5
            )
            if result.returncode == 0:
                return candidate
        except:
            continue

    return "python3"


def find_bridge_script():
    """Find the CLI bridge server script"""
    # Check multiple locations
    candidates = [
        Path.home() / ".eversale" / "engine" / "cli_bridge_server.py",
        Path.home() / ".eversale" / "cli_bridge_server.py",
        Path(__file__).parent.parent.parent / "agent-backend" / "cli_bridge_server.py",
        Path("/mnt/c/ev29/agent-backend/cli_bridge_server.py"),
    ]

    for candidate in candidates:
        if candidate.exists():
            return str(candidate)

    # If not found, use inline server
    return None


def start_inline_server():
    """Start an inline WebSocket server if bridge script not found"""
    global server_running

    import asyncio

    try:
        import websockets
    except ImportError:
        return False, "websockets not installed"

    async def handle_connection(websocket):
        """Simple echo handler for testing"""
        try:
            async for message in websocket:
                # Echo back with success
                response = {
                    "type": "RESULT",
                    "payload": {"success": True, "message": "Server running (inline mode)"}
                }
                await websocket.send(json.dumps(response))
        except:
            pass

    async def run_server():
        global server_running
        server_running = True
        async with websockets.serve(handle_connection, BRIDGE_HOST, BRIDGE_PORT):
            while server_running:
                await asyncio.sleep(0.1)

    # Run in background thread
    def run_async():
        asyncio.run(run_server())

    thread = threading.Thread(target=run_async, daemon=True)
    thread.start()

    return True, BRIDGE_PORT


def start_server():
    """Start the CLI bridge server"""
    global bridge_process, server_running

    # Check if already running
    if is_port_in_use(BRIDGE_PORT):
        log_error(f"Port {BRIDGE_PORT} already in use - server may be running")
        return {"success": True, "port": BRIDGE_PORT, "status": "already_running"}

    # Find bridge script
    bridge_script = find_bridge_script()

    if bridge_script:
        # Start external process
        python_exe = find_python()

        try:
            # Start bridge server as subprocess
            bridge_process = subprocess.Popen(
                [python_exe, bridge_script],
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE,
                cwd=str(Path(bridge_script).parent),
                # Detach from parent on Unix
                start_new_session=True if os.name != 'nt' else False,
                creationflags=subprocess.CREATE_NEW_PROCESS_GROUP if os.name == 'nt' else 0
            )

            # Wait a moment for server to start
            time.sleep(1)

            # Check if it started successfully
            if bridge_process.poll() is None:
                server_running = True
                log_error(f"Bridge server started with PID {bridge_process.pid}")
                return {"success": True, "port": BRIDGE_PORT, "pid": bridge_process.pid}
            else:
                stderr = bridge_process.stderr.read().decode()
                log_error(f"Bridge server failed to start: {stderr}")
                return {"success": False, "error": f"Server failed to start: {stderr}"}

        except Exception as e:
            log_error(f"Error starting bridge server: {e}")
            return {"success": False, "error": str(e)}
    else:
        # Try inline server
        log_error("Bridge script not found, trying inline server")
        success, result = start_inline_server()
        if success:
            return {"success": True, "port": result, "mode": "inline"}
        else:
            return {"success": False, "error": result}


def stop_server():
    """Stop the CLI bridge server"""
    global bridge_process, server_running

    server_running = False

    if bridge_process:
        try:
            # Try graceful termination first
            if os.name == 'nt':
                bridge_process.terminate()
            else:
                os.killpg(os.getpgid(bridge_process.pid), signal.SIGTERM)

            # Wait for termination
            bridge_process.wait(timeout=5)
            log_error(f"Bridge server stopped (PID {bridge_process.pid})")

        except subprocess.TimeoutExpired:
            # Force kill if needed
            if os.name == 'nt':
                bridge_process.kill()
            else:
                os.killpg(os.getpgid(bridge_process.pid), signal.SIGKILL)
            log_error("Bridge server force killed")

        except Exception as e:
            log_error(f"Error stopping bridge server: {e}")
            return {"success": False, "error": str(e)}
        finally:
            bridge_process = None

    return {"success": True, "status": "stopped"}


def main():
    """Main message loop"""
    log_error("Native host started")

    while True:
        message = get_message()

        if message is None:
            # Chrome disconnected
            log_error("Chrome disconnected, stopping server")
            stop_server()
            break

        action = message.get("action", "")
        log_error(f"Received action: {action}")

        if action == "start":
            result = start_server()
            send_message(result)

        elif action == "stop":
            result = stop_server()
            send_message(result)

        elif action == "status":
            send_message({
                "success": True,
                "running": server_running,
                "port": BRIDGE_PORT if server_running else None
            })

        elif action == "ping":
            send_message({"success": True, "pong": True})

        else:
            send_message({"success": False, "error": f"Unknown action: {action}"})

    log_error("Native host exiting")


if __name__ == "__main__":
    main()
