/**
 * WorkflowBuilder - Converts recordings to workflow format
 * Builds DOM maps and workflow definitions from recorded steps
 */
class WorkflowBuilder {
  constructor() {
    this.targetIdCounter = 0;
  }

  /**
   * Build workflow from recorded steps
   * @param {Array} steps - Recorded steps
   * @param {string} name - Workflow name
   * @param {string} description - Workflow description
   * @returns {Object} Workflow and DOM map
   */
  buildFromRecording(steps, name = '', description = '') {
    if (!steps || steps.length === 0) {
      throw new Error('No steps provided');
    }

    // Build DOM map from all steps
    const domMap = this.buildDomMap(steps);

    // Build workflow structure
    const workflow = {
      id: this.generateId(),
      name: name || 'Untitled Workflow',
      description: description || '',
      urlPattern: this.extractUrlPattern(steps[0].url),
      steps: steps.map(s => this.convertStep(s, domMap)),
      domMapId: domMap.id,
      createdAt: Date.now(),
      updatedAt: Date.now(),
      version: '1.0.0'
    };

    return { workflow, domMap };
  }

  /**
   * Build DOM map from recorded steps
   * @param {Array} steps - Recorded steps
   * @returns {Object} DOM map
   */
  buildDomMap(steps) {
    const targets = new Map();

    steps.forEach(step => {
      // Create unique key for this element
      const key = this.createElementKey(step);

      if (!targets.has(key)) {
        const targetId = `target_${this.targetIdCounter++}`;

        targets.set(key, {
          id: targetId,
          selectors: step.selectorCandidates,
          semantic: step.semantic,
          url: step.url,
          firstSeen: step.timestamp
        });
      }
    });

    return {
      id: this.generateId(),
      urlPattern: this.extractUrlPattern(steps[0].url),
      targets: Array.from(targets.values()),
      createdAt: Date.now(),
      updatedAt: Date.now()
    };
  }

  /**
   * Convert recorded step to workflow step
   * @param {Object} recordedStep - Recorded step
   * @param {Object} domMap - DOM map
   * @returns {Object} Workflow step
   */
  convertStep(recordedStep, domMap) {
    // Find matching target in DOM map
    const targetId = this.findTargetId(recordedStep, domMap);

    // Build workflow step based on type
    const step = {
      id: recordedStep.id,
      type: recordedStep.type,
      targetId,
      params: this.convertParams(recordedStep.type, recordedStep.params),
      url: recordedStep.url,
      timestamp: recordedStep.timestamp
    };

    // Add validation rules
    step.validation = this.buildValidation(recordedStep);

    return step;
  }

  /**
   * Find target ID in DOM map for a recorded step
   * @param {Object} recordedStep - Recorded step
   * @param {Object} domMap - DOM map
   * @returns {string} Target ID
   */
  findTargetId(recordedStep, domMap) {
    const key = this.createElementKey(recordedStep);

    const target = domMap.targets.find(t => {
      // Match by semantic features
      return (
        t.semantic.domPath === recordedStep.semantic.domPath ||
        this.selectorsMatch(t.selectors, recordedStep.selectorCandidates)
      );
    });

    return target ? target.id : 'unknown';
  }

  /**
   * Check if selectors match between targets
   * @param {Object} selectors1 - First selector set
   * @param {Object} selectors2 - Second selector set
   * @returns {boolean} True if any selectors match
   */
  selectorsMatch(selectors1, selectors2) {
    // Check if any CSS selectors match
    const css1 = selectors1.css || [];
    const css2 = selectors2.css || [];

    for (const s1 of css1) {
      if (css2.includes(s1)) {
        return true;
      }
    }

    // Check if any XPath selectors match
    const xpath1 = selectors1.xpath || [];
    const xpath2 = selectors2.xpath || [];

    for (const s1 of xpath1) {
      if (xpath2.includes(s1)) {
        return true;
      }
    }

    return false;
  }

  /**
   * Create unique key for element
   * @param {Object} step - Recorded step
   * @returns {string} Element key
   */
  createElementKey(step) {
    const { semantic } = step;

    // Use combination of domPath and primary attributes
    const parts = [
      semantic.domPath,
      semantic.tag,
      semantic.role,
      semantic.attrs?.id || '',
      semantic.attrs?.name || '',
      semantic.text.slice(0, 50)
    ];

    return parts.join('::');
  }

  /**
   * Convert recorded params to workflow params
   * @param {string} type - Step type
   * @param {Object} params - Recorded params
   * @returns {Object} Workflow params
   */
  convertParams(type, params) {
    const converted = { ...params };

    // Simplify params based on type
    switch (type) {
      case 'click':
        return {
          button: params.button || 0,
          modifiers: {
            alt: params.altKey || false,
            ctrl: params.ctrlKey || false,
            shift: params.shiftKey || false,
            meta: params.metaKey || false
          }
        };

      case 'type':
        return {
          value: params.value || '',
          clear: true // Default to clearing before typing
        };

      case 'select':
        return {
          value: params.value,
          text: params.selectedText
        };

      case 'check':
        return {
          checked: params.checked
        };

      case 'scroll':
        return {
          x: params.scrollX || 0,
          y: params.scrollY || 0
        };

      case 'navigate':
        return {
          title: params.title,
          referrer: params.referrer
        };

      default:
        return converted;
    }
  }

  /**
   * Build validation rules for a step
   * @param {Object} recordedStep - Recorded step
   * @returns {Object} Validation rules
   */
  buildValidation(recordedStep) {
    const rules = {
      elementVisible: true,
      elementEnabled: true
    };

    // Add type-specific validations
    switch (recordedStep.type) {
      case 'type':
      case 'change':
        rules.expectedValue = recordedStep.params.value;
        break;

      case 'check':
        rules.expectedChecked = recordedStep.params.checked;
        break;

      case 'select':
        rules.expectedValue = recordedStep.params.value;
        break;
    }

    return rules;
  }

  /**
   * Extract URL pattern from full URL
   * @param {string} url - Full URL
   * @returns {string} URL pattern
   */
  extractUrlPattern(url) {
    try {
      const u = new URL(url);

      // For most cases, use origin + pathname
      let pattern = `${u.origin}${u.pathname}`;

      // Remove trailing slash
      if (pattern.endsWith('/') && pattern.length > 1) {
        pattern = pattern.slice(0, -1);
      }

      // Handle dynamic segments (simple heuristic)
      // Replace numbers in path with wildcards
      pattern = pattern.replace(/\/\d+/g, '/*');

      return pattern;
    } catch (err) {
      console.error('[WorkflowBuilder] Failed to parse URL:', err);
      return url;
    }
  }

  /**
   * Generate unique ID
   * @returns {string} UUID
   */
  generateId() {
    return crypto.randomUUID();
  }

  /**
   * Export workflow to JSON
   * @param {Object} workflow - Workflow object
   * @returns {string} JSON string
   */
  exportWorkflow(workflow) {
    return JSON.stringify(workflow, null, 2);
  }

  /**
   * Import workflow from JSON
   * @param {string} json - JSON string
   * @returns {Object} Workflow object
   */
  importWorkflow(json) {
    try {
      return JSON.parse(json);
    } catch (err) {
      throw new Error('Invalid workflow JSON: ' + err.message);
    }
  }
}

// Make available in both content script and popup contexts
if (typeof window !== 'undefined') {
  window.WorkflowBuilder = WorkflowBuilder;
}
