/**
 * Server Manager - Auto-start/stop CLI bridge server
 *
 * Uses Chrome's Native Messaging to control the local Python server.
 * Falls back to manual WebSocket connection if native messaging unavailable.
 *
 * Usage:
 *   const manager = new ServerManager();
 *   await manager.ensureServerRunning();  // Called when recording starts
 *   await manager.stopServerIfIdle();     // Called when recording stops
 */

class ServerManager {
    constructor() {
        this.hostName = 'com.eversale.host';
        this.port = null;
        this.isServerRunning = false;
        this.nativePort = null;
        this.useNativeMessaging = true;
        this.idleTimeout = null;
        this.IDLE_TIMEOUT_MS = 30000; // Stop server after 30s idle
        this.activeWorkflows = 0;
    }

    /**
     * Initialize native messaging connection
     */
    async init() {
        if (!chrome.runtime.connectNative) {
            console.log('[ServerManager] Native messaging not available');
            this.useNativeMessaging = false;
            return false;
        }

        try {
            this.nativePort = chrome.runtime.connectNative(this.hostName);

            this.nativePort.onMessage.addListener((message) => {
                console.log('[ServerManager] Native message:', message);
                this.handleNativeMessage(message);
            });

            this.nativePort.onDisconnect.addListener(() => {
                const error = chrome.runtime.lastError;
                if (error) {
                    console.log('[ServerManager] Native disconnect:', error.message);
                    // Native messaging not configured - fall back
                    if (error.message.includes('not found') ||
                        error.message.includes('not registered')) {
                        this.useNativeMessaging = false;
                    }
                }
                this.nativePort = null;
                this.isServerRunning = false;
            });

            // Test connection
            return await this.ping();

        } catch (error) {
            console.log('[ServerManager] Native messaging error:', error);
            this.useNativeMessaging = false;
            return false;
        }
    }

    /**
     * Handle message from native host
     */
    handleNativeMessage(message) {
        if (message.success && message.port) {
            this.port = message.port;
            this.isServerRunning = true;
        } else if (message.status === 'stopped') {
            this.isServerRunning = false;
            this.port = null;
        }
    }

    /**
     * Send message to native host
     */
    sendNativeMessage(message) {
        return new Promise((resolve, reject) => {
            if (!this.nativePort) {
                reject(new Error('Native port not connected'));
                return;
            }

            // Set up one-time listener for response
            const listener = (response) => {
                this.nativePort.onMessage.removeListener(listener);
                resolve(response);
            };

            this.nativePort.onMessage.addListener(listener);
            this.nativePort.postMessage(message);

            // Timeout after 10s
            setTimeout(() => {
                this.nativePort.onMessage.removeListener(listener);
                reject(new Error('Native message timeout'));
            }, 10000);
        });
    }

    /**
     * Ping native host to check connection
     */
    async ping() {
        try {
            const response = await this.sendNativeMessage({ action: 'ping' });
            return response && response.pong === true;
        } catch {
            return false;
        }
    }

    /**
     * Start the CLI bridge server
     */
    async startServer() {
        console.log('[ServerManager] Starting server...');

        // Clear any idle timeout
        if (this.idleTimeout) {
            clearTimeout(this.idleTimeout);
            this.idleTimeout = null;
        }

        // Already running?
        if (this.isServerRunning) {
            console.log('[ServerManager] Server already running');
            return { success: true, port: this.port };
        }

        // Try native messaging first
        if (this.useNativeMessaging && this.nativePort) {
            try {
                const response = await this.sendNativeMessage({ action: 'start' });
                if (response.success) {
                    this.port = response.port;
                    this.isServerRunning = true;
                    console.log('[ServerManager] Server started via native messaging');
                    return response;
                }
            } catch (error) {
                console.log('[ServerManager] Native start failed:', error);
            }
        }

        // Fall back to checking if server is already running externally
        const isRunning = await this.checkServerRunning();
        if (isRunning) {
            this.isServerRunning = true;
            this.port = 9876;
            console.log('[ServerManager] External server detected');
            return { success: true, port: this.port, external: true };
        }

        // If native messaging not available and server not running, show instructions
        return {
            success: false,
            error: 'Server not running. Please run: python cli_bridge_server.py',
            needsManualStart: true
        };
    }

    /**
     * Stop the CLI bridge server (with delay for potential reuse)
     */
    async stopServer(immediate = false) {
        if (!this.isServerRunning) {
            return { success: true, status: 'not_running' };
        }

        // Don't stop if there are active workflows
        if (this.activeWorkflows > 0 && !immediate) {
            console.log('[ServerManager] Not stopping - active workflows:', this.activeWorkflows);
            return { success: true, status: 'busy' };
        }

        if (!immediate) {
            // Set idle timeout instead of immediate stop
            console.log('[ServerManager] Setting idle timeout...');
            if (this.idleTimeout) {
                clearTimeout(this.idleTimeout);
            }
            this.idleTimeout = setTimeout(() => {
                this.stopServer(true);
            }, this.IDLE_TIMEOUT_MS);
            return { success: true, status: 'idle_timeout_set' };
        }

        console.log('[ServerManager] Stopping server...');

        // Try native messaging
        if (this.useNativeMessaging && this.nativePort) {
            try {
                const response = await this.sendNativeMessage({ action: 'stop' });
                if (response.success) {
                    this.isServerRunning = false;
                    this.port = null;
                    console.log('[ServerManager] Server stopped via native messaging');
                    return response;
                }
            } catch (error) {
                console.log('[ServerManager] Native stop failed:', error);
            }
        }

        // Mark as not running (external server will keep running)
        this.isServerRunning = false;
        return { success: true, status: 'disconnected' };
    }

    /**
     * Check if server is running by attempting WebSocket connection
     */
    async checkServerRunning() {
        return new Promise((resolve) => {
            const ws = new WebSocket(`ws://localhost:9876`);
            const timeout = setTimeout(() => {
                ws.close();
                resolve(false);
            }, 2000);

            ws.onopen = () => {
                clearTimeout(timeout);
                ws.close();
                resolve(true);
            };

            ws.onerror = () => {
                clearTimeout(timeout);
                resolve(false);
            };
        });
    }

    /**
     * Ensure server is running before recording/execution
     */
    async ensureServerRunning() {
        this.activeWorkflows++;

        if (this.isServerRunning) {
            // Cancel any pending idle timeout
            if (this.idleTimeout) {
                clearTimeout(this.idleTimeout);
                this.idleTimeout = null;
            }
            return { success: true, port: this.port };
        }

        return await this.startServer();
    }

    /**
     * Mark workflow as complete, potentially stopping server
     */
    async workflowComplete() {
        this.activeWorkflows = Math.max(0, this.activeWorkflows - 1);

        if (this.activeWorkflows === 0) {
            // Set idle timeout to stop server
            return await this.stopServer(false);
        }

        return { success: true, status: 'still_active' };
    }

    /**
     * Get current status
     */
    getStatus() {
        return {
            serverRunning: this.isServerRunning,
            port: this.port,
            nativeMessaging: this.useNativeMessaging,
            activeWorkflows: this.activeWorkflows
        };
    }

    /**
     * Cleanup on extension unload
     */
    async cleanup() {
        if (this.idleTimeout) {
            clearTimeout(this.idleTimeout);
        }
        await this.stopServer(true);
        if (this.nativePort) {
            this.nativePort.disconnect();
        }
    }
}

// Singleton instance
let serverManagerInstance = null;

function getServerManager() {
    if (!serverManagerInstance) {
        serverManagerInstance = new ServerManager();
    }
    return serverManagerInstance;
}

// Export for use in extension
if (typeof module !== 'undefined' && module.exports) {
    module.exports = { ServerManager, getServerManager };
}
