class SemanticExtractor {
  constructor() {
    // OPTIMIZED: Reuse SelectorGenerator instance
    this.selectorGenerator = new SelectorGenerator();

    // OPTIMIZED: Static aria attributes list
    this.ariaAttrs = [
      'aria-label',
      'aria-labelledby',
      'aria-describedby',
      'aria-role',
      'aria-checked',
      'aria-selected',
      'aria-expanded'
    ];

    // OPTIMIZED: Static common attributes list
    this.commonAttrs = ['id', 'name', 'type', 'class', 'href', 'src', 'alt', 'title'];
  }

  extract(element) {
    // OPTIMIZED: Single getBoundingClientRect call
    const rect = element.getBoundingClientRect();

    return {
      role: element.getAttribute('role') || this.inferRole(element),
      text: this.extractText(element),
      attrs: this.extractAttrs(element),
      domPath: this.selectorGenerator.buildDomPath(element),
      bbox: {
        x: Math.round(rect.x),
        y: Math.round(rect.y),
        w: Math.round(rect.width),
        h: Math.round(rect.height)
      },
      tag: element.tagName.toLowerCase(),
      visible: this.isVisible(element, rect),
      placeholder: element.placeholder || null,
      value: this.extractValue(element)
    };
  }

  extractText(element) {
    let text = '';

    // For input elements, get placeholder or label
    if (element.tagName === 'INPUT' || element.tagName === 'TEXTAREA') {
      text = element.placeholder || element.getAttribute('aria-label') || '';
    } else {
      // OPTIMIZED: Direct access to childNodes with early exit
      const children = element.childNodes;
      const parts = [];

      for (let i = 0; i < children.length && i < 10; i++) { // OPTIMIZED: Limit children
        const node = children[i];
        if (node.nodeType === Node.TEXT_NODE) {
          parts.push(node.textContent);
        }
      }

      text = parts.join(' ').trim();

      // Fallback to full text content if no direct text
      if (!text) {
        text = element.textContent || '';
      }
    }

    return text.trim().slice(0, 256);
  }

  extractAttrs(element) {
    const attrs = {};

    // Extract data-* attributes
    for (const key in element.dataset) {
      attrs[`data-${key}`] = element.dataset[key];
    }

    // Extract aria-* attributes
    this.ariaAttrs.forEach(attr => {
      const value = element.getAttribute(attr);
      if (value) {
        attrs[attr] = value;
      }
    });

    // Extract common identifying attributes
    this.commonAttrs.forEach(attr => {
      const value = element.getAttribute(attr);
      if (value) {
        attrs[attr] = value;
      }
    });

    return attrs;
  }

  // OPTIMIZED: Static role map
  inferRole(element) {
    const tag = element.tagName.toLowerCase();

    // OPTIMIZED: Switch statement instead of object lookup for common tags
    switch (tag) {
      case 'a': return 'link';
      case 'button': return 'button';
      case 'input': return this.inferInputRole(element);
      case 'textarea': return 'textbox';
      case 'select': return 'combobox';
      case 'img': return 'img';
      case 'nav': return 'navigation';
      case 'header': return 'banner';
      case 'footer': return 'contentinfo';
      case 'main': return 'main';
      case 'aside': return 'complementary';
      case 'section': return 'region';
      case 'article': return 'article';
      case 'form': return 'form';
      case 'table': return 'table';
      case 'ul':
      case 'ol': return 'list';
      case 'li': return 'listitem';
      case 'h1':
      case 'h2':
      case 'h3':
      case 'h4':
      case 'h5':
      case 'h6': return 'heading';
      default: return 'generic';
    }
  }

  // OPTIMIZED: Switch statement for input roles
  inferInputRole(element) {
    const type = element.type?.toLowerCase() || 'text';

    switch (type) {
      case 'button':
      case 'submit':
      case 'reset': return 'button';
      case 'checkbox': return 'checkbox';
      case 'radio': return 'radio';
      case 'range': return 'slider';
      case 'search': return 'searchbox';
      case 'number': return 'spinbutton';
      case 'email':
      case 'tel':
      case 'url':
      case 'text':
      case 'password':
      default: return 'textbox';
    }
  }

  // OPTIMIZED: Pass rect to avoid second getBoundingClientRect call
  isVisible(element, rect) {
    // OPTIMIZED: Single getComputedStyle call
    const style = window.getComputedStyle(element);

    return (
      style.display !== 'none' &&
      style.visibility !== 'hidden' &&
      style.opacity !== '0' &&
      rect.width > 0 &&
      rect.height > 0
    );
  }

  extractValue(element) {
    const tag = element.tagName.toLowerCase();

    if (tag === 'input' || tag === 'textarea') {
      return element.value || null;
    }

    if (tag === 'select') {
      return element.options[element.selectedIndex]?.value || null;
    }

    if (element.hasAttribute('contenteditable')) {
      return element.textContent || null;
    }

    return null;
  }
}

// Make available in content script context
if (typeof window !== 'undefined') {
  window.SemanticExtractor = SemanticExtractor;
}
