class SelectorGenerator {
  constructor() {
    // OPTIMIZED: Cache for selector computations
    this.cache = new WeakMap();
  }

  generateAll(element) {
    // OPTIMIZED: Check cache first
    if (this.cache.has(element)) {
      return this.cache.get(element);
    }

    const result = {
      css: this.generateCssSelectors(element),
      xpath: this.generateXpathSelectors(element)
    };

    // Cache the result
    this.cache.set(element, result);
    return result;
  }

  generateCssSelectors(element) {
    const selectors = [];

    // 1. ID selector - most specific
    if (element.id && /^[a-zA-Z]/.test(element.id)) {
      selectors.push(`#${CSS.escape(element.id)}`);
    }

    // 2. data-testid - common in modern apps
    if (element.dataset.testid) {
      selectors.push(`[data-testid="${CSS.escape(element.dataset.testid)}"]`);
    }

    // 3. data-test attribute
    if (element.dataset.test) {
      selectors.push(`[data-test="${CSS.escape(element.dataset.test)}"]`);
    }

    // 4. aria-label - semantic and accessible
    const ariaLabel = element.getAttribute('aria-label');
    if (ariaLabel) {
      selectors.push(`[aria-label="${CSS.escape(ariaLabel)}"]`);
    }

    // 5. name attribute - for form elements
    if (element.name) {
      const tag = element.tagName.toLowerCase();
      selectors.push(`${tag}[name="${CSS.escape(element.name)}"]`);
    }

    // 6. Class-based selector with tag
    if (element.className && typeof element.className === 'string') {
      const classes = element.className.trim().split(/\s+/).filter(c => c.length > 0);
      if (classes.length > 0 && classes.length <= 3) { // OPTIMIZED: Limit classes
        const tag = element.tagName.toLowerCase();
        const classSelector = classes.slice(0, 3).map(c => `.${CSS.escape(c)}`).join('');
        selectors.push(`${tag}${classSelector}`);
      }
    }

    // 7. Full DOM path with nth-of-type (cached)
    const domPath = this.buildDomPath(element);
    if (domPath) {
      selectors.push(domPath);
    }

    // 8. Text content for buttons and links
    const text = element.textContent?.trim();
    if (text && text.length > 0 && text.length < 50) {
      const tag = element.tagName.toLowerCase();
      if (['a', 'button', 'span', 'div'].includes(tag)) {
        selectors.push(`${tag}:has-text("${CSS.escape(text)}")`);
      }
    }

    return selectors;
  }

  generateXpathSelectors(element) {
    const xpaths = [];

    // 1. ID-based XPath
    if (element.id) {
      xpaths.push(`//*[@id="${element.id}"]`);
    }

    // 2. data-testid XPath
    if (element.dataset.testid) {
      xpaths.push(`//*[@data-testid="${element.dataset.testid}"]`);
    }

    // 3. Full XPath from root
    const fullPath = this.buildXPath(element);
    if (fullPath) {
      xpaths.push(fullPath);
    }

    // 4. Text-based XPath for clickable elements
    const text = element.textContent?.trim();
    if (text && text.length > 0 && text.length < 50) {
      const tag = element.tagName.toLowerCase();
      if (['a', 'button', 'span'].includes(tag)) {
        xpaths.push(`//${tag}[contains(text(), "${text}")]`);
      }
    }

    return xpaths;
  }

  // OPTIMIZED: Limit path depth and cache intermediate results
  buildDomPath(element) {
    const path = [];
    let current = element;
    let depth = 0;
    const maxDepth = 10; // OPTIMIZED: Limit depth for performance

    while (current && current.nodeType === Node.ELEMENT_NODE && depth < maxDepth) {
      let selector = current.tagName.toLowerCase();

      // Add nth-of-type if not unique
      if (current.parentElement) {
        // OPTIMIZED: Cache sibling lookup
        const siblings = Array.from(current.parentElement.children).filter(
          child => child.tagName === current.tagName
        );

        if (siblings.length > 1) {
          const index = siblings.indexOf(current) + 1;
          selector += `:nth-of-type(${index})`;
        }
      }

      path.unshift(selector);

      // Stop at body to keep paths reasonable
      if (current.tagName.toLowerCase() === 'body') {
        break;
      }

      current = current.parentElement;
      depth++;
    }

    return path.join(' > ');
  }

  // OPTIMIZED: Limit path depth
  buildXPath(element) {
    const path = [];
    let current = element;
    let depth = 0;
    const maxDepth = 10; // OPTIMIZED: Limit depth

    while (current && current.nodeType === Node.ELEMENT_NODE && depth < maxDepth) {
      const tag = current.tagName.toLowerCase();

      if (current.parentElement) {
        const siblings = Array.from(current.parentElement.children).filter(
          child => child.tagName === current.tagName
        );

        if (siblings.length > 1) {
          const index = siblings.indexOf(current) + 1;
          path.unshift(`${tag}[${index}]`);
        } else {
          path.unshift(tag);
        }
      } else {
        path.unshift(tag);
      }

      // Stop at body
      if (tag === 'body') {
        break;
      }

      current = current.parentElement;
      depth++;
    }

    return '/' + path.join('/');
  }
}

// Make available in content script context
if (typeof window !== 'undefined') {
  window.SelectorGenerator = SelectorGenerator;
}
