class Recorder {
  constructor() {
    this.isRecording = false;
    this.sessionId = null;
    this.steps = [];
    this.selectorGenerator = new SelectorGenerator();
    this.semanticExtractor = new SemanticExtractor();

    // Bind event handlers
    this.handleClick = this.handleClick.bind(this);
    this.handleInput = this.handleInput.bind(this);
    this.handleChange = this.handleChange.bind(this);
    this.handleScroll = this.handleScroll.bind(this);
    this.handleSubmit = this.handleSubmit.bind(this);
    this.handleKeydown = this.handleKeydown.bind(this);

    // Debounce timers
    this.scrollTimer = null;
    this.inputTimer = null;
    this.inputValue = null;
    this.inputElement = null;
  }

  start(sessionId) {
    if (this.isRecording) return;

    this.isRecording = true;
    this.sessionId = sessionId;
    this.steps = [];

    // OPTIMIZED: Use capture phase with passive listeners where possible
    document.addEventListener('click', this.handleClick, { capture: true, passive: false });
    document.addEventListener('input', this.handleInput, { capture: true, passive: true });
    document.addEventListener('change', this.handleChange, { capture: true, passive: true });
    document.addEventListener('scroll', this.handleScroll, { capture: true, passive: true });
    document.addEventListener('submit', this.handleSubmit, { capture: true, passive: false });
    document.addEventListener('keydown', this.handleKeydown, { capture: true, passive: true });

    // Record initial page load
    this.recordPageLoad();
  }

  stop() {
    if (!this.isRecording) return [];

    this.isRecording = false;

    // Remove event listeners
    document.removeEventListener('click', this.handleClick, { capture: true });
    document.removeEventListener('input', this.handleInput, { capture: true });
    document.removeEventListener('change', this.handleChange, { capture: true });
    document.removeEventListener('scroll', this.handleScroll, { capture: true });
    document.removeEventListener('submit', this.handleSubmit, { capture: true });
    document.removeEventListener('keydown', this.handleKeydown, { capture: true });

    const recordedSteps = [...this.steps];
    this.steps = [];
    this.sessionId = null;

    return recordedSteps;
  }

  handleClick(event) {
    if (!this.isRecording) return;

    const element = event.target;

    // Skip recording clicks on the extension UI itself
    if (this.isExtensionElement(element)) return;

    const step = this.buildRecordedStep('click', element, {
      button: event.button,
      altKey: event.altKey,
      ctrlKey: event.ctrlKey,
      shiftKey: event.shiftKey,
      metaKey: event.metaKey,
      clientX: event.clientX,
      clientY: event.clientY
    });

    this.addStep(step);
  }

  handleInput(event) {
    if (!this.isRecording) return;

    const element = event.target;
    this.inputElement = element;
    this.inputValue = element.value;

    // OPTIMIZED: Debounce with longer timeout to reduce CPU
    clearTimeout(this.inputTimer);
    this.inputTimer = setTimeout(() => {
      if (this.inputElement) {
        const step = this.buildRecordedStep('type', this.inputElement, {
          value: this.inputValue,
          inputType: event.inputType
        });
        this.addStep(step);
        this.inputElement = null;
        this.inputValue = null;
      }
    }, 750); // Increased from 500ms to 750ms
  }

  handleChange(event) {
    if (!this.isRecording) return;

    const element = event.target;
    const tag = element.tagName.toLowerCase();

    let step;

    if (tag === 'select') {
      step = this.buildRecordedStep('select', element, {
        value: element.value,
        selectedIndex: element.selectedIndex,
        selectedText: element.options[element.selectedIndex]?.text
      });
    } else if (element.type === 'checkbox') {
      step = this.buildRecordedStep('check', element, {
        checked: element.checked
      });
    } else if (element.type === 'radio') {
      step = this.buildRecordedStep('check', element, {
        checked: element.checked,
        value: element.value
      });
    } else {
      step = this.buildRecordedStep('change', element, {
        value: element.value
      });
    }

    this.addStep(step);
  }

  handleScroll(event) {
    if (!this.isRecording) return;

    // OPTIMIZED: Throttle scroll events more aggressively
    clearTimeout(this.scrollTimer);
    this.scrollTimer = setTimeout(() => {
      const step = this.buildRecordedStep('scroll', document.documentElement, {
        scrollX: window.scrollX,
        scrollY: window.scrollY,
        scrollHeight: document.documentElement.scrollHeight,
        scrollWidth: document.documentElement.scrollWidth
      });

      this.addStep(step);
    }, 500); // Increased from 300ms to 500ms
  }

  handleSubmit(event) {
    if (!this.isRecording) return;

    const element = event.target;

    const step = this.buildRecordedStep('submit', element, {
      action: element.action,
      method: element.method
    });

    this.addStep(step);
  }

  handleKeydown(event) {
    if (!this.isRecording) return;

    // Only record special keys
    const specialKeys = ['Enter', 'Tab', 'Escape', 'ArrowUp', 'ArrowDown', 'ArrowLeft', 'ArrowRight'];

    if (specialKeys.includes(event.key)) {
      const step = this.buildRecordedStep('keypress', event.target, {
        key: event.key,
        code: event.code,
        altKey: event.altKey,
        ctrlKey: event.ctrlKey,
        shiftKey: event.shiftKey,
        metaKey: event.metaKey
      });

      this.addStep(step);
    }
  }

  recordPageLoad() {
    const step = {
      id: this.generateId(),
      type: 'navigate',
      url: location.href,
      timestamp: Date.now(),
      selectorCandidates: { css: [], xpath: [] },
      semantic: {
        role: 'document',
        text: document.title,
        attrs: {},
        domPath: 'html',
        bbox: { x: 0, y: 0, w: window.innerWidth, h: window.innerHeight },
        tag: 'html',
        visible: true
      },
      params: {
        title: document.title,
        referrer: document.referrer
      }
    };

    this.addStep(step);
  }

  buildRecordedStep(type, element, params) {
    return {
      id: this.generateId(),
      type,
      url: location.href,
      timestamp: Date.now(),
      selectorCandidates: this.selectorGenerator.generateAll(element),
      semantic: this.semanticExtractor.extract(element),
      params
    };
  }

  addStep(step) {
    this.steps.push(step);

    // Send to background script - non-blocking
    chrome.runtime.sendMessage({
      type: 'RECORDED_STEP',
      step,
      sessionId: this.sessionId
    }).catch(() => {
      // Silently fail - background may not be ready
    });
  }

  // OPTIMIZED: Early return with cached check
  isExtensionElement(element) {
    // Quick check on element itself
    if (element.dataset?.eversaleExtension === 'true') return true;

    // Check parents with depth limit
    let current = element.parentElement;
    let depth = 0;
    const maxDepth = 10; // Limit traversal depth

    while (current && depth < maxDepth) {
      if (current.dataset?.eversaleExtension === 'true') return true;
      current = current.parentElement;
      depth++;
    }
    return false;
  }

  generateId() {
    return crypto.randomUUID();
  }
}

// Make available in content script context
if (typeof window !== 'undefined') {
  window.Recorder = Recorder;
}
