/**
 * DomMapStore - IndexedDB storage for DOM maps and workflows
 * Stores element mappings and workflow definitions for replay
 */
class DomMapStore {
  constructor() {
    this.dbName = 'eversale-dom-maps';
    this.dbVersion = 1;
    this.db = null;
  }

  /**
   * Open IndexedDB connection
   * @returns {Promise<IDBDatabase>} Database instance
   */
  async open() {
    if (this.db) {
      return this.db;
    }

    return new Promise((resolve, reject) => {
      const req = indexedDB.open(this.dbName, this.dbVersion);

      req.onupgradeneeded = (event) => {
        const db = event.target.result;

        // Create domMaps object store
        if (!db.objectStoreNames.contains('domMaps')) {
          const domMapStore = db.createObjectStore('domMaps', { keyPath: 'id' });
          domMapStore.createIndex('urlPattern', 'urlPattern', { unique: false });
          domMapStore.createIndex('createdAt', 'createdAt', { unique: false });
        }

        // Create workflows object store
        if (!db.objectStoreNames.contains('workflows')) {
          const workflowStore = db.createObjectStore('workflows', { keyPath: 'id' });
          workflowStore.createIndex('name', 'name', { unique: false });
          workflowStore.createIndex('urlPattern', 'urlPattern', { unique: false });
          workflowStore.createIndex('createdAt', 'createdAt', { unique: false });
        }

        // Create recordings object store for temporary storage
        if (!db.objectStoreNames.contains('recordings')) {
          const recordingStore = db.createObjectStore('recordings', { keyPath: 'id' });
          recordingStore.createIndex('createdAt', 'createdAt', { unique: false });
        }
      };

      req.onsuccess = (event) => {
        this.db = event.target.result;
        resolve(this.db);
      };

      req.onerror = (event) => {
        reject(new Error('Failed to open IndexedDB: ' + event.target.error));
      };
    });
  }

  /**
   * Save DOM map to storage
   * @param {Object} domMap - DOM map object
   * @returns {Promise<string>} Saved DOM map ID
   */
  async saveDomMap(domMap) {
    const db = await this.open();

    return new Promise((resolve, reject) => {
      const transaction = db.transaction(['domMaps'], 'readwrite');
      const store = transaction.objectStore('domMaps');

      const request = store.put(domMap);

      request.onsuccess = () => {
        resolve(domMap.id);
      };

      request.onerror = (event) => {
        reject(new Error('Failed to save DOM map: ' + event.target.error));
      };
    });
  }

  /**
   * Get DOM map by URL pattern
   * @param {string} urlPattern - URL pattern to match
   * @returns {Promise<Object|null>} DOM map or null
   */
  async getDomMap(urlPattern) {
    const db = await this.open();

    return new Promise((resolve, reject) => {
      const transaction = db.transaction(['domMaps'], 'readonly');
      const store = transaction.objectStore('domMaps');
      const index = store.index('urlPattern');

      const request = index.get(urlPattern);

      request.onsuccess = (event) => {
        resolve(event.target.result || null);
      };

      request.onerror = (event) => {
        reject(new Error('Failed to get DOM map: ' + event.target.error));
      };
    });
  }

  /**
   * Get DOM map by ID
   * @param {string} id - DOM map ID
   * @returns {Promise<Object|null>} DOM map or null
   */
  async getDomMapById(id) {
    const db = await this.open();

    return new Promise((resolve, reject) => {
      const transaction = db.transaction(['domMaps'], 'readonly');
      const store = transaction.objectStore('domMaps');

      const request = store.get(id);

      request.onsuccess = (event) => {
        resolve(event.target.result || null);
      };

      request.onerror = (event) => {
        reject(new Error('Failed to get DOM map: ' + event.target.error));
      };
    });
  }

  /**
   * Save workflow to storage
   * @param {Object} workflow - Workflow object
   * @returns {Promise<string>} Saved workflow ID
   */
  async saveWorkflow(workflow) {
    const db = await this.open();

    return new Promise((resolve, reject) => {
      const transaction = db.transaction(['workflows'], 'readwrite');
      const store = transaction.objectStore('workflows');

      const request = store.put(workflow);

      request.onsuccess = () => {
        resolve(workflow.id);
      };

      request.onerror = (event) => {
        reject(new Error('Failed to save workflow: ' + event.target.error));
      };
    });
  }

  /**
   * Get workflow by ID
   * @param {string} id - Workflow ID
   * @returns {Promise<Object|null>} Workflow or null
   */
  async getWorkflow(id) {
    const db = await this.open();

    return new Promise((resolve, reject) => {
      const transaction = db.transaction(['workflows'], 'readonly');
      const store = transaction.objectStore('workflows');

      const request = store.get(id);

      request.onsuccess = (event) => {
        resolve(event.target.result || null);
      };

      request.onerror = (event) => {
        reject(new Error('Failed to get workflow: ' + event.target.error));
      };
    });
  }

  /**
   * Get all workflows
   * @returns {Promise<Array>} Array of workflows
   */
  async getAllWorkflows() {
    const db = await this.open();

    return new Promise((resolve, reject) => {
      const transaction = db.transaction(['workflows'], 'readonly');
      const store = transaction.objectStore('workflows');

      const request = store.getAll();

      request.onsuccess = (event) => {
        resolve(event.target.result || []);
      };

      request.onerror = (event) => {
        reject(new Error('Failed to get workflows: ' + event.target.error));
      };
    });
  }

  /**
   * Delete workflow by ID
   * @param {string} id - Workflow ID
   * @returns {Promise<void>}
   */
  async deleteWorkflow(id) {
    const db = await this.open();

    return new Promise((resolve, reject) => {
      const transaction = db.transaction(['workflows'], 'readwrite');
      const store = transaction.objectStore('workflows');

      const request = store.delete(id);

      request.onsuccess = () => {
        resolve();
      };

      request.onerror = (event) => {
        reject(new Error('Failed to delete workflow: ' + event.target.error));
      };
    });
  }

  /**
   * Save temporary recording
   * @param {Object} recording - Recording object
   * @returns {Promise<string>} Recording ID
   */
  async saveRecording(recording) {
    const db = await this.open();

    return new Promise((resolve, reject) => {
      const transaction = db.transaction(['recordings'], 'readwrite');
      const store = transaction.objectStore('recordings');

      const request = store.put(recording);

      request.onsuccess = () => {
        resolve(recording.id);
      };

      request.onerror = (event) => {
        reject(new Error('Failed to save recording: ' + event.target.error));
      };
    });
  }

  /**
   * Get recording by ID
   * @param {string} id - Recording ID
   * @returns {Promise<Object|null>} Recording or null
   */
  async getRecording(id) {
    const db = await this.open();

    return new Promise((resolve, reject) => {
      const transaction = db.transaction(['recordings'], 'readonly');
      const store = transaction.objectStore('recordings');

      const request = store.get(id);

      request.onsuccess = (event) => {
        resolve(event.target.result || null);
      };

      request.onerror = (event) => {
        reject(new Error('Failed to get recording: ' + event.target.error));
      };
    });
  }

  /**
   * Delete recording by ID
   * @param {string} id - Recording ID
   * @returns {Promise<void>}
   */
  async deleteRecording(id) {
    const db = await this.open();

    return new Promise((resolve, reject) => {
      const transaction = db.transaction(['recordings'], 'readwrite');
      const store = transaction.objectStore('recordings');

      const request = store.delete(id);

      request.onsuccess = () => {
        resolve();
      };

      request.onerror = (event) => {
        reject(new Error('Failed to delete recording: ' + event.target.error));
      };
    });
  }
}

// Make available in both content script and popup contexts
if (typeof window !== 'undefined') {
  window.DomMapStore = DomMapStore;
}
