/**
 * CLI Bridge Client - Connect to local CLI server
 *
 * Handles:
 * - WebSocket connection to CLI
 * - Sending workflows for execution
 * - Syncing DOM maps
 * - Receiving execution results
 */

class CLIBridge {
    constructor(host = 'localhost', port = 9876) {
        this.url = `ws://${host}:${port}`;
        this.ws = null;
        this.isConnected = false;
        this.pendingRequests = new Map(); // requestId -> {resolve, reject}
        this.reconnectAttempts = 0;
        this.maxReconnectAttempts = 5;
        this.listeners = new Map(); // event -> callbacks
    }

    /**
     * Connect to CLI bridge server
     */
    async connect() {
        return new Promise((resolve, reject) => {
            try {
                this.ws = new WebSocket(this.url);

                this.ws.onopen = () => {
                    console.log('[CLI Bridge] Connected to CLI');
                    this.isConnected = true;
                    this.reconnectAttempts = 0;
                    this.emit('connected');
                    resolve(true);
                };

                this.ws.onclose = () => {
                    console.log('[CLI Bridge] Disconnected from CLI');
                    this.isConnected = false;
                    this.emit('disconnected');
                    this.attemptReconnect();
                };

                this.ws.onerror = (error) => {
                    console.error('[CLI Bridge] Connection error:', error);
                    this.emit('error', error);
                    reject(error);
                };

                this.ws.onmessage = (event) => {
                    this.handleMessage(event.data);
                };

            } catch (error) {
                reject(error);
            }
        });
    }

    /**
     * Disconnect from CLI
     */
    disconnect() {
        if (this.ws) {
            this.ws.close();
            this.ws = null;
        }
        this.isConnected = false;
    }

    /**
     * Attempt to reconnect after disconnect
     */
    attemptReconnect() {
        if (this.reconnectAttempts >= this.maxReconnectAttempts) {
            console.log('[CLI Bridge] Max reconnect attempts reached');
            return;
        }

        this.reconnectAttempts++;
        const delay = Math.min(1000 * Math.pow(2, this.reconnectAttempts), 30000);

        console.log(`[CLI Bridge] Reconnecting in ${delay}ms (attempt ${this.reconnectAttempts})`);

        setTimeout(() => {
            this.connect().catch(() => {
                // Will retry via onclose handler
            });
        }, delay);
    }

    /**
     * Send message and wait for response
     */
    async send(type, payload) {
        if (!this.isConnected) {
            throw new Error('Not connected to CLI');
        }

        return new Promise((resolve, reject) => {
            const requestId = crypto.randomUUID();

            // Set timeout for response
            const timeout = setTimeout(() => {
                this.pendingRequests.delete(requestId);
                reject(new Error('Request timeout'));
            }, 60000); // 60s timeout

            this.pendingRequests.set(requestId, {
                resolve: (result) => {
                    clearTimeout(timeout);
                    resolve(result);
                },
                reject: (error) => {
                    clearTimeout(timeout);
                    reject(error);
                }
            });

            const message = JSON.stringify({
                type,
                payload,
                requestId
            });

            this.ws.send(message);
        });
    }

    /**
     * Handle incoming message
     */
    handleMessage(data) {
        try {
            const msg = JSON.parse(data);

            // Check if this is a response to a pending request
            if (msg.requestId && this.pendingRequests.has(msg.requestId)) {
                const { resolve, reject } = this.pendingRequests.get(msg.requestId);
                this.pendingRequests.delete(msg.requestId);

                if (msg.type === 'ERROR') {
                    reject(new Error(msg.payload.error));
                } else {
                    resolve(msg.payload);
                }
                return;
            }

            // Handle broadcast messages
            this.emit('message', msg);

        } catch (error) {
            console.error('[CLI Bridge] Failed to parse message:', error);
        }
    }

    /**
     * Execute a workflow via CLI
     */
    async executeWorkflow(workflow, variables = {}) {
        return this.send('EXECUTE_WORKFLOW', { workflow, variables });
    }

    /**
     * Sync DOM map to CLI
     */
    async syncDomMap(domMap) {
        return this.send('SYNC_DOM_MAP', { domMap });
    }

    /**
     * Check if CLI is ready
     */
    async getStatus() {
        return this.send('GET_STATUS', {});
    }

    /**
     * List available workflows
     */
    async listWorkflows() {
        const result = await this.send('LIST_WORKFLOWS', {});
        return result.workflows || [];
    }

    /**
     * Event handling
     */
    on(event, callback) {
        if (!this.listeners.has(event)) {
            this.listeners.set(event, []);
        }
        this.listeners.get(event).push(callback);
    }

    off(event, callback) {
        if (this.listeners.has(event)) {
            const callbacks = this.listeners.get(event);
            const index = callbacks.indexOf(callback);
            if (index > -1) {
                callbacks.splice(index, 1);
            }
        }
    }

    emit(event, data) {
        if (this.listeners.has(event)) {
            for (const callback of this.listeners.get(event)) {
                callback(data);
            }
        }
    }
}

// Export for use in extension
if (typeof module !== 'undefined' && module.exports) {
    module.exports = { CLIBridge };
}
