// OPTIMIZED: Lazy-initialized recorder instance
let recorder = null;

// OPTIMIZED: Initialize recorder only when needed
function initRecorder() {
  if (!recorder) {
    recorder = new window.Recorder();
  }
  return recorder;
}

// OPTIMIZED: Handle messages with minimal overhead
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  switch (message.type) {
    case 'START_RECORDING':
      handleStartRecording(message, sendResponse);
      return true;

    case 'STOP_RECORDING':
      handleStopRecording(sendResponse);
      return true;

    case 'GET_RECORDER_STATE':
      handleGetRecorderState(sendResponse);
      return true;

    case 'PING':
      sendResponse({ ok: true });
      return true;

    default:
      sendResponse({ error: 'Unknown message type' });
      return true;
  }
});

function handleStartRecording(message, sendResponse) {
  try {
    // Initialize recorder on-demand
    const rec = initRecorder();
    rec.start(message.sessionId);
    addRecordingIndicator();

    sendResponse({
      ok: true,
      sessionId: message.sessionId,
      url: window.location.href
    });
  } catch (err) {
    sendResponse({
      ok: false,
      error: err.message
    });
  }
}

function handleStopRecording(sendResponse) {
  try {
    if (!recorder) {
      throw new Error('Recorder not initialized');
    }

    const steps = recorder.stop();
    removeRecordingIndicator();

    sendResponse({
      ok: true,
      steps,
      url: window.location.href
    });
  } catch (err) {
    sendResponse({
      ok: false,
      error: err.message
    });
  }
}

function handleGetRecorderState(sendResponse) {
  if (!recorder) {
    sendResponse({
      isRecording: false,
      sessionId: null,
      stepCount: 0
    });
    return;
  }

  sendResponse({
    isRecording: recorder.isRecording,
    sessionId: recorder.sessionId,
    stepCount: recorder.steps.length
  });
}

// OPTIMIZED: Minimal DOM manipulation for recording indicator
function addRecordingIndicator() {
  removeRecordingIndicator();

  const indicator = document.createElement('div');
  indicator.id = 'eversale-recording-indicator';
  indicator.setAttribute('data-eversale-extension', 'true');

  // Inline styles for performance
  indicator.style.cssText = 'position:fixed;top:10px;right:10px;background:#ef4444;color:#fff;padding:8px 16px;border-radius:20px;font-family:system-ui,-apple-system,sans-serif;font-size:14px;font-weight:600;z-index:2147483647;box-shadow:0 2px 8px rgba(0,0,0,.2);display:flex;align-items:center;gap:8px;pointer-events:none';

  const dot = document.createElement('span');
  dot.style.cssText = 'width:8px;height:8px;background:#fff;border-radius:50%;animation:eversale-pulse 1.5s ease-in-out infinite';

  // Add animation keyframes
  if (!document.getElementById('eversale-indicator-style')) {
    const style = document.createElement('style');
    style.id = 'eversale-indicator-style';
    style.textContent = '@keyframes eversale-pulse{0%,100%{opacity:1}50%{opacity:.4}}';
    document.head.appendChild(style);
  }

  indicator.appendChild(dot);
  indicator.appendChild(document.createTextNode('Recording'));

  document.body.appendChild(indicator);
}

function removeRecordingIndicator() {
  const indicator = document.getElementById('eversale-recording-indicator');
  if (indicator) {
    indicator.remove();
  }
}

// OPTIMIZED: Non-blocking notification to background
chrome.runtime.sendMessage({
  type: 'CONTENT_SCRIPT_READY',
  url: window.location.href
}).catch(() => {
  // Silently fail - background may not be ready
});
