// Import server manager and CLI bridge
importScripts('lib/serverManager.js');
importScripts('lib/cliBridge.js');

// Recording state - minimal memory footprint
let currentRecording = null;
let recordedSteps = [];
let contentScriptReady = new Map();
let serverManager = null;

// OPTIMIZED: Lazy initialization
async function initServerManager() {
  if (!serverManager) {
    serverManager = getServerManager();
    await serverManager.init();
  }
  return serverManager;
}

// OPTIMIZED: Single message handler with minimal overhead
chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
  switch (msg.type) {
    case 'START_RECORDING':
      handleStartRecording(sender, sendResponse);
      return true;

    case 'STOP_RECORDING':
      handleStopRecording(sendResponse);
      return true;

    case 'RECORDED_STEP':
      handleRecordedStep(msg, sendResponse);
      return true;

    case 'GET_RECORDING_STATUS':
      handleGetRecordingStatus(sendResponse);
      return true;

    case 'CONTENT_SCRIPT_READY':
      handleContentScriptReady(sender, sendResponse);
      return true;

    case 'GET_RECORDED_STEPS':
      handleGetRecordedSteps(sendResponse);
      return true;

    case 'GET_SERVER_STATUS':
      handleGetServerStatus(sendResponse);
      return true;

    case 'EXECUTE_WORKFLOW':
      handleExecuteWorkflow(msg, sendResponse);
      return true;

    default:
      sendResponse({ error: 'Unknown message type' });
      return true;
  }
});

async function handleStartRecording(sender, sendResponse) {
  try {
    const manager = await initServerManager();

    // Auto-start server
    const serverResult = await manager.ensureServerRunning();

    // Get active tab
    let tabId;
    if (sender.tab) {
      tabId = sender.tab.id;
    } else {
      const tabs = await chrome.tabs.query({ active: true, currentWindow: true });
      if (tabs.length === 0) {
        throw new Error('No active tab found');
      }
      tabId = tabs[0].id;
    }

    // Initialize recording state
    const sessionId = generateSessionId();
    currentRecording = {
      tabId,
      sessionId,
      startedAt: Date.now(),
      url: sender.tab?.url || '',
      serverPort: serverResult.port || null
    };
    recordedSteps = [];

    // Send START_RECORDING to content script
    const response = await chrome.tabs.sendMessage(tabId, {
      type: 'START_RECORDING',
      sessionId
    });

    sendResponse({
      ok: true,
      sessionId,
      tabId,
      serverRunning: serverResult.success || serverResult.external,
      serverPort: serverResult.port
    });
  } catch (err) {
    sendResponse({
      ok: false,
      error: err.message
    });
  }
}

async function handleStopRecording(sendResponse) {
  try {
    if (!currentRecording) {
      throw new Error('No recording in progress');
    }

    // Send STOP_RECORDING to content script
    await chrome.tabs.sendMessage(currentRecording.tabId, {
      type: 'STOP_RECORDING'
    });

    // Build result
    const result = {
      ok: true,
      sessionId: currentRecording.sessionId,
      steps: recordedSteps,
      duration: Date.now() - currentRecording.startedAt,
      url: currentRecording.url,
      stepCount: recordedSteps.length
    };

    // Save workflow to storage for later execution
    if (recordedSteps.length > 0) {
      await saveLastWorkflow(result);
    }

    // Clear recording state
    currentRecording = null;

    // Mark workflow complete
    if (serverManager) {
      await serverManager.workflowComplete();
    }

    sendResponse(result);
  } catch (err) {
    sendResponse({
      ok: false,
      error: err.message
    });
  }
}

async function saveLastWorkflow(recordingResult) {
  try {
    const workflow = {
      id: recordingResult.sessionId,
      name: `Recording ${new Date().toLocaleString()}`,
      description: `Recorded from ${new URL(recordingResult.url).hostname}`,
      urlPattern: new URL(recordingResult.url).origin + '/*',
      steps: recordingResult.steps.map(step => ({
        id: step.id,
        action: step.type,
        selector: step.selectorCandidates?.css?.[0],
        targetId: step.semantic?.domPath ? hashCode(step.semantic.domPath) : null,
        value: step.params?.value,
        timeout: 30000
      })),
      createdAt: Date.now(),
      updatedAt: Date.now()
    };

    // Build DOM map from steps
    const domMap = {
      id: hashCode(workflow.urlPattern),
      urlPattern: workflow.urlPattern,
      targets: {},
      updatedAt: Date.now()
    };

    recordingResult.steps.forEach(step => {
      if (step.semantic) {
        const targetId = hashCode(step.semantic.domPath || step.id);
        domMap.targets[targetId] = {
          id: targetId,
          urlPattern: workflow.urlPattern,
          selectorCandidates: step.selectorCandidates,
          semantic: step.semantic,
          lastVerifiedAt: Date.now(),
          failureCount: 0,
          successCount: 1
        };
      }
    });

    await chrome.storage.local.set({
      lastWorkflow: workflow,
      lastDomMap: domMap,
      lastRecording: recordingResult
    });
  } catch (err) {
    // Silently fail - not critical
  }
}

// Simple hash function
function hashCode(str) {
  let hash = 0;
  for (let i = 0; i < str.length; i++) {
    const char = str.charCodeAt(i);
    hash = ((hash << 5) - hash) + char;
    hash = hash & hash;
  }
  return Math.abs(hash).toString(16).padStart(8, '0');
}

function handleRecordedStep(msg, sendResponse) {
  if (!currentRecording) {
    sendResponse({ ok: false, error: 'Not recording' });
    return;
  }

  if (msg.sessionId !== currentRecording.sessionId) {
    sendResponse({ ok: false, error: 'Session ID mismatch' });
    return;
  }

  recordedSteps.push(msg.step);
  sendResponse({ ok: true });
}

function handleGetRecordingStatus(sendResponse) {
  const serverStatus = serverManager ? serverManager.getStatus() : { serverRunning: false };

  sendResponse({
    isRecording: !!currentRecording,
    sessionId: currentRecording?.sessionId || null,
    stepCount: recordedSteps.length,
    tabId: currentRecording?.tabId || null,
    startedAt: currentRecording?.startedAt || null,
    serverRunning: serverStatus.serverRunning,
    serverPort: serverStatus.port
  });
}

async function handleGetServerStatus(sendResponse) {
  const manager = await initServerManager();
  const status = manager.getStatus();
  const isRunning = await manager.checkServerRunning();

  sendResponse({
    ...status,
    actuallyRunning: isRunning
  });
}

async function handleExecuteWorkflow(msg, sendResponse) {
  try {
    const manager = await initServerManager();

    // Ensure server is running
    const serverResult = await manager.ensureServerRunning();
    if (!serverResult.success && !serverResult.external) {
      throw new Error('CLI server not available');
    }

    // Connect to CLI bridge and execute
    const bridge = new CLIBridge();
    await bridge.connect();

    // Get workflow from message or storage
    let workflow = msg.workflow;
    let domMap = msg.domMap;

    if (!workflow) {
      const stored = await chrome.storage.local.get(['lastWorkflow', 'lastDomMap']);
      workflow = stored.lastWorkflow;
      domMap = stored.lastDomMap;
    }

    if (!workflow) {
      throw new Error('No workflow to execute');
    }

    // Sync DOM map if available
    if (domMap) {
      await bridge.syncDomMap(domMap);
    }

    // Execute workflow
    const result = await bridge.executeWorkflow(workflow, msg.variables || {});

    // Cleanup
    bridge.disconnect();
    await manager.workflowComplete();

    sendResponse({
      ok: true,
      result
    });
  } catch (err) {
    sendResponse({
      ok: false,
      error: err.message
    });
  }
}

function handleContentScriptReady(sender, sendResponse) {
  if (sender.tab) {
    contentScriptReady.set(sender.tab.id, true);
  }
  sendResponse({ ok: true });
}

function handleGetRecordedSteps(sendResponse) {
  sendResponse({
    ok: true,
    steps: recordedSteps,
    sessionId: currentRecording?.sessionId || null
  });
}

function generateSessionId() {
  return `session_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
}

// OPTIMIZED: Tab event handlers with minimal overhead
chrome.tabs.onUpdated.addListener((tabId, changeInfo) => {
  if (changeInfo.status === 'complete') {
    contentScriptReady.delete(tabId);
  }
});

chrome.tabs.onRemoved.addListener((tabId) => {
  contentScriptReady.delete(tabId);

  if (currentRecording && currentRecording.tabId === tabId) {
    currentRecording = null;
    recordedSteps = [];

    if (serverManager) {
      serverManager.workflowComplete();
    }
  }
});

// Cleanup on suspend
chrome.runtime.onSuspend.addListener(async () => {
  if (serverManager) {
    await serverManager.cleanup();
  }
});
